---@meta
--- polyfill.lua
--- Polyfill for Lua 5.1 and 5.2 compatibility.

_G.polyfill = {
	version = 3,
	info = function()
		print("JPXS polyfill is loaded!")
		print("Version: " .. polyfill.version)
	end,
}

function table.find(table, value)
	for k, v in pairs(table) do
		if v == value then
			return k
		end
	end
end

function table.dictLength(dict)
	local len = 0
	for _, _ in pairs(dict) do
		len = len + 1
	end
	return len
end

---@param table table
---@return any[]
function table.flat(table)
	local result = {}
	for _, value in pairs(table) do
		if type(value) == "table" then
			for _, v in pairs(value) do
				table.insert(result, v)
			end
		else
			table.insert(result, value)
		end
	end
	return result
end

---get the keys of a table.
---@param tbl table The table to get the keys from.
function table.keys(tbl)
	local keys = {}
	for k, _ in pairs(tbl) do
		table.insert(keys, k)
	end
	return keys
end

---get the values of a table.
---@param tbl table The table to get the values from.
function table.values(tbl)
	local values = {}
	for _, v in pairs(tbl) do
		table.insert(values, v)
	end
	return values
end

--Clone a table
---@param object table The object to copy.
---@param destination? table The table to copy the object to.
---@return table copy The copied object.
function table.clone(object, destination)
	destination = destination or {}
	for key, value in pairs(object) do
		if type(value) == "table" then
			destination[key] = table.clone(value)
		else
			destination[key] = value
		end
	end
	return destination
end

---Check if a string starts with another string.
---@param self string Added to avoid typing issues.
---@param start string The string to check against.
---@return boolean startsWith Whether this string starts with the other.
function string.startsWith(self, start)
	return self:sub(1, #start) == start
end

---Check if a string ends with another string.
---@param self string Added to avoid typing issues.
---@param ending string The string to check against.
---@return boolean endsWith Whether this string ends with the other.
function string.endsWith(self, ending)
	return ending == "" or self:sub(-#ending) == ending
end

---Split a string by its whitespace into lines of maximum length.
---@param self string Added to avoid typing issues.
---@param maxLen integer The maximum length of every line.
---@return string[] lines The split lines.
function string.splitMaxLen(self, maxLen)
	local lines = {}
	local line

	local _, _ = self:gsub("(%s*)(%S+)", function(spc, word)
		if not line or #line + #spc + #word > maxLen then
			table.insert(lines, line)
			line = word
		else
			line = line .. spc .. word
		end
	end)

	table.insert(lines, line)
	return lines
end

---Split a string into tokens using a separator character.
---@param self string Added to avoid typing issues.
---@param sep string The separator character.
---@return string[] fields The split tokens.
function string.split(self, sep)
	sep = sep or ":"
	local fields = {}
	local pattern = string.format("([^%s]+)", sep)
	local _, _ = self:gsub(pattern, function(c)
		fields[#fields + 1] = c
	end)
	return fields
end

---Trim whitespace before and after a string.
---@param self string Added to avoid typing issues.
---@return string trimmed The trimmed string.
---@return integer count
function string.trim(self)
	return self:gsub("^%s*(.-)%s*$", "%1")
end

function decimalToHex(decimal)
	local hexChars = "0123456789ABCDEF"
	local hexResult = ""

	while decimal > 0 do
		local remainder = decimal % 16
		hexResult = hexChars:sub(remainder + 1, remainder + 1) .. hexResult
		decimal = math.floor(decimal / 16)
	end

	if hexResult == "" then
		hexResult = "0"
	end

	return hexResult
end

function ipToBytes(ip)
	local tab = string.split(ip, ".")
	local str = "0x"
	for ind, val in ipairs(tab) do
		str = str .. decimalToHex(tonumber(val))
	end
	return tonumber(str)
end

---@param addr integer The address to read from.
---@return Vector vec The vector read from the address.
function memory.readVector(addr)
	local x = memory.readFloat(addr)
	local y = memory.readFloat(addr + 4)
	local z = memory.readFloat(addr + 8)
	local vec = Vector(x, y, z)
	return vec
end

---@param addr integer The address to write to.
---@param Vector Vector The vector to write to the address.
---@return nil
function memory.writeVector(addr, Vector)
	memory.writeFloat(addr, Vector.x)
	memory.writeFloat(addr + 4, Vector.y)
	memory.writeFloat(addr + 8, Vector.z)
end

---@param str string
---@param list string[]
function filterTableStartsWith(str, list)
	for _, v in ipairs(list) do
		if v:startsWith(str) then
			return v
		end
	end
	return str
end
